#include "StMouse.h"
#include <assert.h>
#include <Process.h>
#include <stdio.h>

/// <summary>
/// This function receives raw position data from read thread and parses each 
/// coordinate i.e. S,X,Y and Z into position structure and forwards it to the
/// subscriber.
/// </summary>
/// <param name="data">Raw mouse data string</param>
void StMouse::SetPosition(std::string data)
{
	try {
		const char* arr = data.c_str();
		const char* ms;
		int temp;
		StPosition newPosition;

		// Find and parse S component of data.
		ms = strstr(arr, "S");
		if (ms != NULL) {
			if (sscanf_s(ms, "S%d", &temp) == 1) {
				newPosition.S = temp;
			}
		}

		// Find and parse X component of data.
		ms = strstr(arr, "X");
		if (ms != NULL) {
			if (sscanf_s(ms, "X%d", &temp) == 1) {
				newPosition.X = temp;
			}
		}

		// Find and parse Y component of data.
		ms = strstr(arr, "Y");
		if (ms != NULL) {
			if (sscanf_s(ms, "Y%d", &temp) == 1) {
				newPosition.Y = temp;
			}
		}

		// Find and parse Z component of data.
		ms = strstr(arr, "Z");
		if (ms != NULL) {
			if (sscanf_s(ms, "Z%d", &temp) == 1) {
				newPosition.Z = temp;
			}
		}

		// Forward new position data to subscriber.
		if (dCallback != 0)
			dCallback(newPosition); 
	}
	catch (...) {
		// Do nothing.
	}
}

/// <summary>
/// This function is used to invalidates the handle when no longer needed.
/// </summary>
/// <param name="hHandle">Handle to invalidate</param>
void StMouse::InvalidateHandle(HANDLE& hHandle)
{
	hHandle = INVALID_HANDLE_VALUE;
}

/// <summary>
/// This function tries to close the handle and then invalidates it.
/// </summary>
/// <param name="hHandle">Handle to be closed</param>
void StMouse::CloseAndCleanHandle(HANDLE& hHandle)
{
	BOOL abRet = CloseHandle(hHandle);
	if (!abRet)
	{
		assert(0);
	}
	InvalidateHandle(hHandle);
}

/// <summary>
/// This function starts a separate thread to read data from FTDI USB device.
/// </summary>
void StMouse::StartListenerThread()
{
	m_hThreadStarted = CreateEvent(0, 0, 0, 0);
	assert(m_hThreadStarted && "Could not create event");

	m_hThread = (HANDLE)_beginthreadex(0, 0, StMouse::ThreadFn, (void*)this, 0, 0);

	DWORD dwWait = WaitForSingleObject(m_hThreadStarted, INFINITE);
	assert(dwWait == WAIT_OBJECT_0);
	CloseHandle(m_hThreadStarted);
	InvalidateHandle(m_hThreadStarted);
	m_abIsConnected = true;
}

/// <summary>
/// This function closes the read thread and removes the subscriber callback.
/// Its automatically called when Close() is called.
/// </summary>
/// <returns>Returns S_OK if successful, E_FAIL if fails.</returns>
HRESULT StMouse::UnInit()
{
	HRESULT hr = S_OK;
	try
	{
		m_abIsConnected = false;
		CloseAndCleanHandle(m_hThread);
	}
	catch (...)
	{
		assert(0);
		hr = E_FAIL;
	}
	if (SUCCEEDED(hr))
		m_eState = SS_UnInit;

	dCallback = nullptr;

	return hr;
}

/// <summary>
/// StMouse Constructor.
/// </summary>
StMouse::StMouse()
{
	InvalidateHandle(m_hThread);
	InvalidateHandle(m_hThreadStarted);

	InitLock();
	m_eState = SS_UnInit;
}

/// <summary>
/// StMouse Desconstructor.
/// </summary>
StMouse::~StMouse()
{
	m_eState = SS_Unknown;
	DelLock();
}

/// <summary>
/// Open the device and creates a handle which will be used for subsequent accesses.
/// </summary>
/// <param name="iDevice">Index of the device to open. Indices are 0 based.</param>
/// <returns>TRUE if successful, otherwise returns FALSE. </returns>
BOOL StMouse::Connect(int iDevice)
{
	LastCallStatus = FT_Open(iDevice, &m_ftHandle);
	if (LastCallStatus == FT_OK) {
		// FT_Open OK, use ftHandle to access device
		return Setup();
	}
	else {
		return FALSE;
	}
}

/// <summary>
/// Open the specified device and creates a handle that will be used for subsequent accesses. 
/// The device is specified by its serial number.
/// </summary>
/// <param name="pvArg1">A pointer to a null-terminated string that represents the serial 
/// number of the device</param>
/// <returns>TRUE if successful, otherwise returns FALSE. </returns>
BOOL StMouse::ConnectBySerialNumber(PVOID pvArg1)
{
	LastCallStatus = FT_OpenEx(pvArg1, FT_OPEN_BY_SERIAL_NUMBER, &m_ftHandle);
	if (LastCallStatus == FT_OK) {
		// FT_Open OK, use ftHandle to access device
		return Setup();
	}
	else {
		// FT_Open failed
		return FALSE;
	}
}

/// <summary>
/// Sets up initial parameters for the device such data/stop bits, partiy, flow control and 
/// timeouts.
/// </summary>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
BOOL StMouse::Setup() {
	// Set baud rate.
	BOOL ftStatus = SetBaudRate(9600);
	if (!ftStatus) {
		return FALSE;
	}

	// Set data characteristics - Data bits, Stop bits, Parity
	ftStatus = SetDataCharacteristics(FT_BITS_8, FT_STOP_BITS_1, FT_PARITY_NONE);
	if (!ftStatus) {
		return FALSE;
	}

	// Set flow control - set RTS/CTS flow control
	ftStatus = SetFlowControl(FT_FLOW_RTS_CTS, 0x11, 0x13);
	if (!ftStatus) {
		return FALSE;
	}

	// Old: Set read timeout to 5 seconds, write timeout to infinite
	// New : Set timeouts to 1 millisecond.
	ftStatus = SetTimeouts(1, 1);
	if (!ftStatus) {
		return FALSE;
	}

	// Set In/Out request size to 64 bytes only.
	ftStatus = SetUSBParameters(64, 64);
	if (!ftStatus) {
		return FALSE;
	}

	StartListenerThread();

	return TRUE;
}

/// <summary>
/// This function sets the data characteristics for the device.
/// </summary>
/// <param name="uWordLength">Number of bits per word - must be FT_BITS_8 or FT_BITS_7</param>
/// <param name="uStopBits">Number of stop bits - must be FT_STOP_BITS_1 or FT_STOP_BITS_2.</param>
/// <param name="uParity">Parity - must be FT_PARITY_NONE, FT_PARITY_ODD, FT_PARITY_EVEN, 
/// FT_PARITY_MARK or FT_PARITY SPACE</param>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
BOOL StMouse::SetDataCharacteristics(UCHAR uWordLength, UCHAR uStopBits, UCHAR uParity)
{
	LastCallStatus = FT_SetDataCharacteristics(m_ftHandle, uWordLength, uStopBits, uParity);
	if (LastCallStatus == FT_OK) {
		// Data characteristics set.
		return TRUE;
	}
	else {
		// Failed to set data characteristics.
		return FALSE;
	}
}

/// <summary>
/// This function sets the flow control for the device.
/// </summary>
/// <param name="usFlowControl">Must be one of FT_FLOW_NONE, FT_FLOW_RTS_CTS, FT_FLOW_DTR_DSR or FT_FLOW_XON_XOFF.</param>
/// <param name="uXon">Character used to signal Xon. Only used if flow control is FT_FLOW_XON_XOFF.</param>
/// <param name="uXoff">Character used to signal Xoff. Only used if flow control is FT_FLOW_XON_XOFF</param>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
BOOL StMouse::SetFlowControl(USHORT usFlowControl, UCHAR uXon, UCHAR uXoff)
{
	LastCallStatus = FT_SetFlowControl(m_ftHandle, usFlowControl, uXon, uXoff);
	if (LastCallStatus == FT_OK) {
		// Flow control set.
		return TRUE;
	}
	else {
		// Failed to set flow control.
		return FALSE;
	}
}

/// <summary>
/// This function sets the In/Out USB request transfer size.
/// </summary>
/// <param name="dwInTransferSize">Transfer size for USB IN request.</param>
/// <param name="dwOutTransferSize">Transfer size for USB OUT request.</param>
/// <returns></returns>
BOOL StMouse::SetUSBParameters(DWORD dwInTransferSize, DWORD dwOutTransferSize)
{
	LastCallStatus = FT_SetUSBParameters(m_ftHandle, dwInTransferSize, dwOutTransferSize);
	if (LastCallStatus == FT_OK) {
		// Flow control set.
		return TRUE;
	}
	else {
		// Failed to set flow control.
		return FALSE;
	}
}

/// <summary>
/// This function sets the baud rate for the device.
/// </summary>
/// <param name="dwBaudRate">Baud rate. e.g. 9600, 115200 etc.</param>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
BOOL StMouse::SetBaudRate(DWORD dwBaudRate)
{
	LastCallStatus = FT_SetBaudRate(m_ftHandle, dwBaudRate);
	if (LastCallStatus == FT_OK) {
		// BaudRate changed.
		return TRUE;
	}
	else {
		// Failed to change BaudRate
		return FALSE;
	}
}

/// <summary>
/// This function sets the read and write timeouts for the device.
/// </summary>
/// <param name="dwReadTimeout">Read timeout in milliseconds.</param>
/// <param name="dwWriteTimeout">Write timeout in milliseconds.</param>
/// <returns>TRUE if successful, otherwise returns FALSE.</returns>
BOOL StMouse::SetTimeouts(DWORD dwReadTimeout, DWORD dwWriteTimeout)
{
	LastCallStatus = FT_SetTimeouts(m_ftHandle, dwReadTimeout, dwWriteTimeout);
	if (LastCallStatus == FT_OK) {
		// Timeouts set.
		return TRUE;
	}
	else {
		// Failed to set timeouts.
		return FALSE;
	}
}

/// <summary>
/// Close an open device.
/// </summary>
/// <returns>TRUE if successful, otherwise returns FALSE. </returns>
BOOL StMouse::Close()
{
	LastCallStatus = FT_Close(m_ftHandle);
	if (LastCallStatus == FT_OK) {
		// Closed.
		return TRUE;
	}
	else {
		// FT_Open failed
		return FALSE;
	}

	// Call Unint anyway.
	UnInit();
}

/// <summary>
/// This function is a utility method to convert FT_STATUS enum to its string representation.
/// </summary>
/// <param name="st">FT_STATUS object to convert.</param>
/// <returns>If successful, returns a FT_STATUS matching string, if fails returns 'UnKnown'.</returns>
std::string StMouse::GetErrorString(FT_STATUS st)
{
	switch (st)
	{
	case FT_OK:
		return "OK";
	case FT_INVALID_HANDLE:
		return "Invalid Handle";
	case FT_DEVICE_NOT_FOUND:
		return "Device Not Found";
	case FT_DEVICE_NOT_OPENED:
		return "Device Not Opened";
	case FT_IO_ERROR:
		return "IO Error";
	case FT_INSUFFICIENT_RESOURCES:
		return "Insufficient Resources";
	case FT_INVALID_PARAMETER:
		return "Invalid Parameter";
	case FT_INVALID_BAUD_RATE:
		return "Invalid Baud Rate";
	case FT_DEVICE_NOT_OPENED_FOR_ERASE:
		return "Device Not Opened For Erase";
	case FT_DEVICE_NOT_OPENED_FOR_WRITE:
		return "Device Not Opened For Write";
	case FT_FAILED_TO_WRITE_DEVICE:
		return "Failed To Write Device";
	case FT_EEPROM_READ_FAILED:
		return "EEPROM Read Failed";
	case FT_EEPROM_WRITE_FAILED:
		return "EEPROM Write Failed";
	case FT_EEPROM_ERASE_FAILED:
		return "EEPROM Erase Failed";
	case FT_EEPROM_NOT_PRESENT:
		return "EEPROM Not Present";
	case FT_EEPROM_NOT_PROGRAMMED:
		return "EEPROM Not Programmed";
	case FT_INVALID_ARGS:
		return "Invalid Arguments";
	case FT_NOT_SUPPORTED:
		return "Not Supported";
	case FT_OTHER_ERROR:
		return "Other Error";
	case FT_DEVICE_LIST_NOT_READY:
		return "Device List Not Ready";
	default:
		break;
	}

	return "UnKnown";
}

/// <summary>
/// Its a thread function that runs in a loop to get FT_GetQueueStatus, If finds any new bytes to read,
/// It keeps calling FT_Read until a <CR> is detected. Once <CR> is received, it forwards the string to
/// SetPosition function.
/// </summary>
/// <param name="pvParam">StMouse object to receive data updates through SetPosition function.</param>
/// <returns></returns>
unsigned __stdcall StMouse::ThreadFn(void* pvParam)
{
	StMouse* apThis = (StMouse*)pvParam;
	bool abContinue = true;
	bool isComplete = true;
	DWORD dwEventMask = 0;

	DWORD RxBytes;
	DWORD BytesReceived;
	char RxBuffer[64] = { '\0' };
	DWORD dwWait;
	SetEvent(apThis->m_hThreadStarted);

	while (abContinue)
	{
		char Temp[64] = { '\0' };
		// Get queue status.
		FT_GetQueueStatus(apThis->m_ftHandle, &RxBytes);
		if (RxBytes > 0) {
			FT_STATUS LastCallStatus = FT_Read(apThis->m_ftHandle, Temp, RxBytes, &BytesReceived);
			if (LastCallStatus == FT_OK && BytesReceived > 0) {
				if (isComplete)
					// If last read was completed with <CR>, copy the new data to the start of buffer.
					strncpy_s(RxBuffer, Temp, sizeof(Temp));
				else
					// If we have not received <CR> yet. Keep appending to the buffer.
					strncat_s(RxBuffer, Temp, BytesReceived);

				if (Temp[BytesReceived - 1] != 0x0D) {
					isComplete = false;
					continue;
				}

				isComplete = true;
				apThis->SetPosition(RxBuffer);
			}

			// If USB device disconnected, break the loop.
			if (LastCallStatus == FT_IO_ERROR) {
				abContinue = false;
				break;
			}
		}
	}
	return 0;
}

/// <summary>
/// This function is used to subscribe to the data updates.
/// </summary>
/// <param name="cB">Callback function of void return type and one input parameter of type StPosition</param>
void StMouse::setDCallback(type_myCallBack cB)
{
	dCallback = cB;
}
